// ------------------ LPShid.c ---------------------
//
// Vaunix Digital Phase Shifter Library for Linux
//
// Version .91, Sept. 19, 2015
//

#include <usb.h>
#include <linux/hid.h>
#include <stdbool.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <pthread.h>
#include <time.h>
#include "LPShid.h"

#define DEBUG_OUT 0  		/* set this to 1 in order to see debugging output, 2 for a ton of output, or 3 for many tons */
#define FALSE 0
#define TRUE !FALSE

#define PACKET_CTRL_LEN 8
#define PACKET_INT_LEN 8
#define INTERFACE 0
#define ENDPOINT_INT_IN 0x82
#define TIMEOUT 500

#define LIBVER "0.91"		// RD: initial version of the Digital Phase Shifter Library

void *brick_handler_function (void *ptr);

/* ----------------------------------------------------------------- */
/* globals we'll be using at runtime */
char errmsg[32]; 			// For the status->string converter

bool bVerbose = FALSE; 		// True to generate debug oriented printf output

bool TestMode = FALSE; 		// if TestMode is true we fake it -- no HW access
							// TestMode can be set to FALSE for production builds

LPSPARAMS lps [MAXDEVICES];	// an array of structures each of which holds the info for a given 
							// device. The DeviceID is the index into the array. Devices may come and go
							// so there is no guarantee that the active elements are contiguous

time_t starttime, currtime;

// product names
char sVNX1[32] = "LPS-802";
char sVNX2[32] = "LPS-123";
char sVNX3[32] = "LPS-402";
char sVNX4[32] = "LPS-202";

// device VID and PID
unsigned short devVID = 0x041f;	 // device VID for Vaunix Devices

unsigned short dev1PID = 0x1240; // device PID for Vaunix LPS-802
unsigned short dev2PID = 0x1241; // device PID for Vaunix LPS-123
unsigned short dev3PID = 0x1242; // device PID for Vaunix LPS-402
unsigned short dev4PID = 0x1243; // device PID for Vaunix LPS-202

/* stuff for the threads */
pthread_t threads[MAXDEVICES];
usb_dev_handle *thread_devhandles[MAXDEVICES];
#define THREAD_IDLE 0
#define THREAD_START 1
#define THREAD_EXIT 3
#define THREAD_DEAD 4
#define THREAD_ERROR -1

// --------------- Device IO support functions ----------------------------

bool CheckDeviceOpen(DEVID deviceID) {
  if (TestMode) return TRUE;	// in test mode all devices are always available
  if ((lps[deviceID].DevStatus & DEV_OPENED) && (deviceID != 0))
    return TRUE;
  else
    return FALSE;
}

// ------------------------------------------------------------------------
bool DevNotLocked(DEVID deviceID) {
  if (TestMode) return TRUE;	// this shouldn't happen, but just in case...
  if (!(lps[deviceID].DevStatus & DEV_LOCKED))
    return TRUE;				// we return TRUE if the device is not locked!
  else
    return FALSE;
}

// ------------------------------------------------------------------------
void LockDev(DEVID deviceID, bool lock) {
  if (TestMode) return;			// this shouldn't happen, but just in case...
  if (lock) {
    lps[deviceID].DevStatus = lps[deviceID].DevStatus | DEV_LOCKED;
    return;
  } else {
    lps[deviceID].DevStatus = lps[deviceID].DevStatus & ~DEV_LOCKED;
    return;
  }
}

/* A function to display the status as string */
char* fnLPS_perror(LPSTATUS status) {
  strcpy(errmsg, "STATUS_OK");
  if (BAD_PARAMETER == status) strcpy(errmsg, "BAD_PARAMETER");
  if (BAD_HID_IO == status) strcpy(errmsg, "BAD_HID_IO");
  if (DEVICE_NOT_READY == status) strcpy(errmsg, "DEVICE_NOT_READY");
  
  // Status returns for DevStatus
  if (INVALID_DEVID == status) strcpy(errmsg, "INVALID_DEVID");
  if (DEV_CONNECTED == status) strcpy(errmsg, "DEV_CONNECTED");
  if (DEV_OPENED == status) strcpy(errmsg, "DEV_OPENED");
  if (SWP_ACTIVE == status) strcpy(errmsg,  "SWP_ACTIVE");
  if (SWP_UP == status) strcpy(errmsg, "SWP_UP");
  if (SWP_REPEAT == status) strcpy(errmsg, "SWP_REPEAT");
  
  return errmsg;
}

char LibVersion[] = LIBVER;
char* fnLPS_LibVersion(void) {
  return LibVersion;
}
  
/* functions based on hid_io.cpp */
bool VNXOpenDevice(DEVID deviceID) {

  if (!(lps[deviceID].DevStatus & DEV_CONNECTED))	// we can't open a device that isn't there!
    return DEVICE_NOT_READY;
  
  if (DEBUG_OUT > 1) printf("Starting thread %d\r\n", deviceID);
  lps[deviceID].thread_command = THREAD_START; 		// open device and start processing
  pthread_create(&threads[deviceID], NULL, brick_handler_function, (void*)deviceID);
  lps[deviceID].DevStatus = lps[deviceID].DevStatus | DEV_OPENED;
  
  return STATUS_OK;
}

void report_data_decode(unsigned char rcvdata[], int tid) {
  int i;
  unsigned long dataval;
  char temp[32];

  if ((DEBUG_OUT > 1) && (rcvdata[0] != 0x4e)) {
    printf("Decoding ");
    for (i=0; i<8; i++)
      printf("%02x ", rcvdata[i]);
    printf("\r\n");
  }

  /* the first byte tells us the type, the second is the data length
     tid is the thread ID it came from so we can stash the value into lps[] */
	 
  // first decode the common cases of a DWORD or WORD value
  dataval = 0;
  if (rcvdata[1] == 2 || rcvdata[1] == 4) {
    for (i=0; i<rcvdata[1]; i++)
      dataval = (dataval<<8) + rcvdata[1+rcvdata[1]-i];
  }
  if ((DEBUG_OUT > 1) && (rcvdata[0] != VNX_PHSTATUS)) printf("Data payload decodes to %ld (%08x)\r\n", dataval, dataval);
  /* now we'll assign it to lps[] */

  // handle the status report
  switch(rcvdata[0]) {
  case VNX_PHSTATUS:
    if (DEBUG_OUT > 2) printf("VNX_PHSTATUS: got one!\r\n");
    if (DevNotLocked(tid)) {
	  // update the phase angle
	  dataval = 0;
	  dataval = rcvdata[5] + (rcvdata[6]<<8);			// phase angle is a WORD
	  lps[tid].PhaseAngle = dataval;
	  
	  // update the profile index
	  if (rcvdata[4] >= 0 && rcvdata[4] < VNX_PROFILE_LENGTH)
				lps[tid].ProfileIndex = rcvdata[4];		// update the profile index after range check
	  if (DEBUG_OUT > 2) printf("  VNX_STATUS reports Phase Angle =%d, Profile Index =%d\r\n", lps[tid].PhaseAngle, lps[tid].ProfileIndex);
	  
	  // update the 1KHz frame counter
	  dataval = 0;
	  dataval = rcvdata[2] + (rcvdata[3]<<8);			// the frame counter is 11 bits wide, stored in a WORD
	  lps[tid].FrameNumber = dataval;  
	  
      if (rcvdata[7] & (SWP_ONCE | SWP_CONTINUOUS))		// are we ramping?
		lps[tid].DevStatus = lps[tid].DevStatus | SWP_ACTIVE;
      else
		lps[tid].DevStatus = lps[tid].DevStatus & ~SWP_ACTIVE;

      // -- fill in the SWP_UP status bit
      if (rcvdata[7] & (SWP_DIRECTION))					// are we ramping downwards?
		lps[tid].DevStatus = lps[tid].DevStatus & ~SWP_UP;
      else
		lps[tid].DevStatus = lps[tid].DevStatus | SWP_UP;
      // -- fill in the continuous sweep bit
      if (rcvdata[7] & (SWP_CONTINUOUS))				// are we in continuous sweep mode?
		lps[tid].DevStatus = lps[tid].DevStatus | SWP_REPEAT;
      else
		lps[tid].DevStatus = lps[tid].DevStatus & ~SWP_REPEAT;
	  // -- fill in the bidirectional sweep bit
	  if (rcvdata[7] & (SWP_BIDIR))						// are we in bi-directional sweep mode?
		lps[tid].DevStatus = lps[tid].DevStatus | SWP_BIDIRECTIONAL;
	  else
		lps[tid].DevStatus = lps[tid].DevStatus & ~SWP_BIDIRECTIONAL;
	  // -- fill in the profile active bit
	  if (rcvdata[7] & (STATUS_PROFILE_ACTIVE))			// is a phase angle profile playing?
		lps[tid].DevStatus = lps[tid].DevStatus | PROFILE_ACTIVE;
	  else
		lps[tid].DevStatus = lps[tid].DevStatus & ~PROFILE_ACTIVE;
	
      if (DEBUG_OUT > 2) printf("  VNX_STATUS: Status=%02x, Frame Counter = %d\r\n", lps[tid].DevStatus, lps[tid].FrameNumber);

    } // if devnotlocked()
    break;

  case VNX_PHASEANGLE:
    if (DEBUG_OUT > 0) printf(" Phase Angle = %d\n", dataval);
    if (DevNotLocked(tid))
	  lps[tid].PhaseAngle = dataval;
    break;
	
  case VNX_FREQUENCY:
    if (DEBUG_OUT > 0) printf(" Working Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
	  lps[tid].Frequency = dataval;
    break;
		
  case VNX_ADWELL:
    if (DEBUG_OUT > 0) printf(" First Section Dwell Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].DwellTime = dataval;
    break;
	
  case VNX_ADWELL2:
    if (DEBUG_OUT > 0) printf(" Second Section Dwell Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].DwellTime2 = dataval;
    break;
	
  case VNX_AIDLE:
    if (DEBUG_OUT > 0) printf(" Idle Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].IdleTime = dataval;
    break;
	
  case VNX_AHOLD:
    if (DEBUG_OUT > 0) printf(" Hold Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].HoldTime = dataval;
    break;
	
  case VNX_ASTART:
    if (DEBUG_OUT > 0) printf(" Ramp Start Phase Angle = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].RampStart = dataval;
    break;

  case VNX_ASTEP:
    if (DEBUG_OUT > 0) printf(" First Section Phase Angle Step Size = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].PhaseStep = dataval;
    break;
	
  case VNX_ASTEP2:
    if (DEBUG_OUT > 0) printf(" Second Section Phase Angle Step Size = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].PhaseStep2 = dataval;
    break;
	
  case VNX_ASTOP:
     if (DEBUG_OUT > 0) printf(" Ramp End Phase Angle = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].RampStop = dataval;
    break;

  case VNX_SWEEP:
    if (DEBUG_OUT > 0) printf(" Ramp Mode = %d\n", rcvdata[2]);
    if (DevNotLocked(tid)) {
 	// not used, same information available in the VNX_STATUS report
    }
    break;

  case VNX_PROFILEIDLE:
    if (DEBUG_OUT > 0) printf(" Profile Idle Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].ProfileIdleTime = dataval;
    break;
	
  case VNX_PROFILEDWELL:
    if (DEBUG_OUT > 0) printf(" Profile Dwell Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].ProfileDwellTime = dataval;
    break;
	
  case VNX_PROFILECOUNT:
    if (DEBUG_OUT > 0) printf(" Profile Count = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].ProfileCount = dataval;
    break;		

  case VNX_SETPROFILE:
	dataval = 0;
	dataval = rcvdata[2] + (rcvdata[3]<<8);				// phase angle is a WORD at byteblock[0] and [1]
	if (DevNotLocked(tid))
	  {
		i = rcvdata[6];
		if ( i >= 0 && i < VNX_PROFILE_LENGTH)			// defend our array
		  lps[tid].Profile[i] = dataval & 0x0000FFFF;	// clip off the unused high word
	  }
	if (DEBUG_OUT > 0) printf(" Profile Element %d = %d\n", i, (dataval & 0x0000FFFF));
	break;
	
  case VNX_MINATTEN:
    if (DEBUG_OUT > 0) printf(" Minimum Phase Angle = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].MinPhase = dataval;
    break;

  case VNX_MAXATTEN:
    if (DEBUG_OUT > 0) printf(" Maximum Phase Angle = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].MaxPhase = dataval;
    break;

  case VNX_MINFREQUENCY:
    if (DEBUG_OUT > 0) printf(" Minimum Operating Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].MinFrequency = dataval;
    break;
	
  case VNX_MAXFREQUENCY:
    if (DEBUG_OUT > 0) printf(" Maximum Operating Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].MaxFrequency = dataval;
    break;	
	
  case VNX_GETSERNUM:
    if (DEBUG_OUT > 0) printf(" Serial Number = %d\n", dataval);
    if (DevNotLocked(tid))
      lps[tid].SerialNumber = dataval;		// NB -- we never use this path!
    break;
  } /* switch */

  return;
}

// ************* The read thread handler for the brick ***********************
void *brick_handler_function (void *threadID) {
  int i, tid;
  tid = (int)threadID;
  struct usb_bus *busses;
  struct usb_bus *bus;
  int usb_status;
  char fullpath[128];
  int retries;
 
  if (DEBUG_OUT > 0) printf("Starting thread for device %d\r\n", tid);
  while ((lps[tid].thread_command >=0) &&
	 (lps[tid].thread_command != THREAD_EXIT)) {
    switch(lps[tid].thread_command) {
    case THREAD_IDLE: /* idle */
      /* this is where we wait for incoming USB data */
      usb_status = -1;
      retries = 50;
      while ((usb_status < 0) && (retries--) && (THREAD_IDLE == lps[tid].thread_command)) {
	usb_status = usb_interrupt_read(thread_devhandles[tid],  // handle
					ENDPOINT_INT_IN, // endpoint
					lps[tid].rcvbuff, // buffer
					PACKET_INT_LEN, // max length
					TIMEOUT);
	if (usb_status < 0) usleep(1000); /* wait 20 ms before trying again */
      }
      //      printf("Thread %d reports %d...\r\n", tid, usb_status);
      if (usb_status >= 0) {
	if (DEBUG_OUT > 1) {
	  printf("Thread %d reports %d...", tid, usb_status);
	  for (i=0; i<usb_status; i++)
	    printf("%02x ", lps[tid].rcvbuff[i]);
	  printf("\r\n");
	}
	/* decode the HID data */
	report_data_decode(lps[tid].rcvbuff, tid);
	if (DEBUG_OUT > 1) printf("Decoded device %d data %02x, decodewatch=%02x\r\n", tid, lps[tid].rcvbuff[0], lps[tid].decodewatch);
	if (lps[tid].decodewatch == lps[tid].rcvbuff[0]) {
	  if (DEBUG_OUT > 0) printf("Clearing decodewatch %02x for thread %d\r\n", lps[tid].decodewatch, tid);
	  lps[tid].decodewatch = 0;
	}
      } else
	if (DEBUG_OUT > 0) perror("THREAD_IDLE");
      break;
    case THREAD_START: /* starting up */
      /* we'll open the device. First we have to locate it */
      if (DEBUG_OUT > 0) printf("Thread %d is looking for the device\r\n", tid);

      usb_find_busses();
      usb_find_devices();
      busses = usb_get_busses();

      lps[tid].thread_command = THREAD_ERROR; /* assume it will fail */
      for (bus = busses; bus; bus = bus->next) {
	if (THREAD_IDLE == lps[tid].thread_command) break;
	struct usb_device *dev;

	for (dev = bus->devices; dev; dev = dev->next) {
	  if (THREAD_IDLE == lps[tid].thread_command) break;
	  if (DEBUG_OUT > 1) printf("Thread %d sez- Vendor: %04x PID: %04x\r\n", tid, dev->descriptor.idVendor, dev->descriptor.idProduct);
	  thread_devhandles[tid] = usb_open(dev);
	  usb_status = usb_get_string_simple(thread_devhandles[tid], dev->descriptor.iSerialNumber, lps[tid].rcvbuff, sizeof(lps[tid].rcvbuff));
	  if (DEBUG_OUT > 1) printf("string %d = [%s] looking to match [%s]\r\n", dev->descriptor.iSerialNumber, lps[tid].rcvbuff, lps[tid].Serialstr);
	  if ((dev->descriptor.idVendor == lps[tid].idVendor) &&
	      (dev->descriptor.idProduct == lps[tid].idProduct) &&
	      (0 == strcmp(lps[tid].rcvbuff, lps[tid].Serialstr))) {
	    /* we found the device. We'll open it */
	    if (DEBUG_OUT > 1) printf("Opening file [%s]\r\n", dev->filename);
	    thread_devhandles[tid] = usb_open(dev);

	    usb_detach_kernel_driver_np(thread_devhandles[tid], 0);

	    usb_status = usb_set_configuration (thread_devhandles[tid], 1);
	    if (DEBUG_OUT > 1) printf ("set configuration: %s\n", usb_status ? "failed" : "passed");

	    usb_status = usb_claim_interface (thread_devhandles[tid], 0);
	    if (DEBUG_OUT > 1) printf ("claim interface: %s\n", usb_status ? "failed" : "passed");

	    lps[tid].thread_command = THREAD_IDLE;
	    break;
	  } else {
	    /* if the device we opened isn't the one we wanted, close it */
	    usb_close(thread_devhandles[tid]);
	  }
	} /* for dev */
      } /* for bus */
      break;
    } /* switch */
  } /* while */
  if (DEBUG_OUT > 0) printf("Exiting thread for device %d because command=%d\r\n", tid, lps[tid].thread_command);
  if (THREAD_EXIT == lps[tid].thread_command)
    usb_close(thread_devhandles[tid]);
  lps[tid].thread_command = THREAD_DEAD;
  pthread_exit(NULL);
}

// -------------- SendReport -------------------------------------------------

bool SendReport(int deviceID, char command, char *pBuffer, int cbBuffer)
{
  int i;
  int send_status;
  int retries;
  // Make sure the buffer that is being passed to us fits
  if (cbBuffer > HR_BLOCKSIZE) {
    // Report too big, don't send!
    return FALSE;
  }

  char Report[8];
  
  if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%x\r\n", command, cbBuffer);
  lps[deviceID].sndbuff[0] = command;		// command to device
  lps[deviceID].sndbuff[1] = cbBuffer;
  lps[deviceID].sndbuff[2] = pBuffer[0];
  lps[deviceID].sndbuff[3] = pBuffer[1];
  lps[deviceID].sndbuff[4] = pBuffer[2];
  lps[deviceID].sndbuff[5] = pBuffer[3];
  lps[deviceID].sndbuff[6] = pBuffer[4];
  lps[deviceID].sndbuff[7] = pBuffer[4];
  if (DEBUG_OUT > 1) {
    printf("SR: ");
    for (i=0; i<8; i++) {
      printf("%02x ", lps[deviceID].sndbuff[i]);
    }
    printf("\r\n");
  }

  /* we have to wait for a file handle to appear */
  retries = 0;
  while ((0 == thread_devhandles[deviceID]) && (retries++ < 10))
    sleep(1);
  /* we have data to write to the device */
  if (DEBUG_OUT > 1) printf("SR: sending the write...\r\n");
  send_status = usb_control_msg(thread_devhandles[deviceID],
				0x21,
				0x09, //HID_REPORT_SET,
				0x200,
				0,
				lps[deviceID].sndbuff,
				PACKET_CTRL_LEN,
				TIMEOUT);

  if (DEBUG_OUT > 1) {
    printf("(status=%d handle=%d)", send_status, thread_devhandles[deviceID]);
    if (send_status < 0) perror("SendReport"); else printf("\r\n");
  }

  return TRUE;
}

// -------------- Protect Profile Element 19 ------------------------------
//
// if this unit needs profile element 19 protection we reset the profile element to 
// the value saved in our host side cache of profile values.

void ProtectProfile(DEVID deviceID)
{
	if (lps[deviceID].NeedsProfileFix && lps[deviceID].Profile[19] != -1)
	{
		char VNX_command[] = {0, 0, 0, 0, 0};

		VNX_command[4] = (char) 19;					// index
		VNX_command[0] = (char) lps[deviceID].Profile[19];		// low byte of 16 bit phase angle
		VNX_command[1] = (char)(lps[deviceID].Profile[19] >> 8);	// high byte of 16 bit phase angle

		SendReport(deviceID, VNX_SETPROFILE | VNX_SET, VNX_command, 5);
	}
}




// ------------ GetParameter ---------------------------------------------
//
// The GetParam argument is the command byte sent to the device to get
// a particular value. The response is picked up by the read thread and
// parsed by it. The parser clears the corresponding event.


bool GetParameterEx(int deviceID, int GetParam, int ex_param)
{
	char VNX_param[5] = {0, 0, 0, 0, 0};
	int timedout;
	int sendcount = 0;

	if (DEBUG_OUT > 0) printf(" sending a GET command = %x\n", (char) GetParam );
	lps[deviceID].decodewatch = (char) GetParam;
	
	// -- the extended parameter is specific to each command --
	if (GetParam == VNX_SETPROFILE | VNX_GET) {
	  VNX_param[4] = (char) ex_param;
	  sendcount = 5;
	}
	
	if (!SendReport(deviceID, (char)GetParam, VNX_param, sendcount)) {
	  return FALSE;
	}

	if (DEBUG_OUT > 0) printf(" SendReport sent a GET command successfully to device %d = %x\n", deviceID, (char) GetParam );
	
	starttime = time(NULL);
	timedout = 0;

	/* wait until the value is decoded or 2 seconds have gone by */
	while ((lps[deviceID].decodewatch > 0) && (0 == timedout)) {
	  if ((time(NULL)-starttime) > 2) timedout = 1;
	}

	return (0 == timedout) ? TRUE : FALSE;
}

bool GetParameter(int deviceID, int GetParam) {
		return GetParameterEx(deviceID, GetParam, 0);
}



// -------------- Get Routines to read device settings --------------------
//
// Note: for these functions deviceID is not checked for validity
//		 since it was already checked in the calling program.

bool GetPhaseAngle(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PHASEANGLE))
    return FALSE;
  return TRUE;
}

bool GetFrequency(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_FREQUENCY))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetIdleTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_AIDLE))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetHoldTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_AHOLD))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetRampStart(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTART))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetRampStep(DEVID deviceID) {	
  if (!GetParameter(deviceID, VNX_ASTEP))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetRampStep2(DEVID deviceID) {	
  if (!GetParameter(deviceID, VNX_ASTEP2))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetRampEnd(DEVID deviceID) {	
  if (!GetParameter(deviceID, VNX_ASTOP))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetDwellTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ADWELL))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetDwellTime2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ADWELL2))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetProfileDwellTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEDWELL))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetProfileCount(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILECOUNT))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetProfileIdleTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEIDLE))
    return FALSE;
  ProtectProfile(deviceID);		// if necessary restore profile element 19
  return TRUE;
}

bool GetProfileElement(DEVID deviceID, int index){
  if (index < 0 || index >= VNX_PROFILE_LENGTH) return FALSE;			// bail out on a bad index...
  
  if (!GetParameterEx(deviceID, VNX_PROFILEIDLE, index)) return FALSE;	// or an I/O failure...
  return TRUE;
}

// not used by LPS devices
bool GetRF_On(DEVID deviceID) {
  return TRUE;
}

/* functions to manage devices, not getting or retrieving data */
/*-------------------------------------------------------------*/

void FindVNXDevices()
{
  bool bFound;
  int hTemp;  		// temporary variable
  int HWType; 		// temporary variable for hardware/model type
  int HWMinPhase;	// temporary variable for default minimum phase angle
  int HWMaxPhase;	// temporary variable for default maximum phase angle in 1 degree units
  int HWPhaseStep;	// temporary variable for default phase step in 1 degree units
  int HWFMinimum;	// temporary variable for default minimum operating frequency in 100KHz units
  int HWFMaximum;	// temporary variable for default maximum operating frequency in 100KHz units
  char HWName[32];  // temporary variable for the hardware model name
  char HWSerial[8]; // temporary holder for the serial number
  struct usb_dev_handle *devhandle;
  struct usb_bus *busses;
  char sendbuff[8];
  char rcvbuff[32];
  int usb_status;
    
  usb_init();
  if (DEBUG_OUT > 2)
    usb_set_debug(3); 	/* if we want lots of debug, let's see the USB output too. */
  else
    usb_set_debug(0);
  usb_find_busses();
  usb_find_devices();
  
  busses = usb_get_busses();
        
  struct usb_bus *bus;
  int c, i, a;
  int send_status, open_status;
     
  // We need to remove devices from our table that are no longer connected ---
  // to do this we clear the "connected" flag for each table entry that is not open initially
  // then, as we find them we re-set the "connected" flag
  // anybody who doesn't have his "connected" flag set at the end is gone - we found it
  // previously but not this time
  
  for (i = 1; i<MAXDEVICES; i++){
    if ((lps[i].SerialNumber != 0) && !(lps[i].DevStatus & DEV_OPENED))
      lps[i].DevStatus = lps[i].DevStatus & ~DEV_CONNECTED; 	
  }

  for (bus = busses; bus; bus = bus->next) {
    struct usb_device *dev;
    
    for (dev = bus->devices; dev; dev = dev->next) {
      if (DEBUG_OUT > 1) printf("Vendor: %04x PID: %04x\r\n", dev->descriptor.idVendor, dev->descriptor.idProduct);
      HWType = 0;
      /* check this device to see if it's one of our devices */
      if ((devVID == dev->descriptor.idVendor) &&
	  (dev1PID == dev->descriptor.idProduct)) HWType = 1;
      if ((devVID == dev->descriptor.idVendor) &&
	  (dev2PID == dev->descriptor.idProduct)) HWType = 2;
	  if ((devVID == dev->descriptor.idVendor) &&
	  (dev3PID == dev->descriptor.idProduct)) HWType = 3;
	  if ((devVID == dev->descriptor.idVendor) &&
	  (dev4PID == dev->descriptor.idProduct)) HWType = 4;
	  
      if (HWType) { /* we like this device and we'll keep it */
	  
	    if (DEBUG_OUT > 1) printf("Opening device %04x:%04x serial %04x type %d\r\n",
			      dev->descriptor.idVendor,
			      dev->descriptor.idProduct,
			      dev->descriptor.iSerialNumber, HWType);
	    devhandle = usb_open(dev);
	    if (DEBUG_OUT > 1)  printf("LPS device found @ address [%s]\r\n", dev->filename);
	      usb_status = usb_get_string_simple(devhandle, dev->descriptor.iSerialNumber, rcvbuff, sizeof(rcvbuff));
	    if (DEBUG_OUT > 1) printf("string %d = [%s]\r\n", dev->descriptor.iSerialNumber, rcvbuff);
	    if (usb_status < 0) strcpy(HWSerial, ""); else strcpy(HWSerial, rcvbuff+3);
	    usb_close(devhandle);
	
		switch(HWType) {
		case 1:						// LPS-802
		  strcpy(HWName, sVNX1);
		  HWMinPhase = 0;
		  HWMaxPhase = 360;
		  HWPhaseStep = 1;
		  HWFMinimum = 40000;		// 4000 MHz
		  HWFMaximum = 80000;		// 8000 MHz
		  break;
		case 2:						// LPS-123
		  strcpy(HWName, sVNX2);
		  HWMinPhase = 0;
		  HWMaxPhase = 360;
		  HWPhaseStep = 1;
		  HWFMinimum = 80000;		// 8000 MHz
		  HWFMaximum = 120000;		// 12000 MHz
		  break;
		case 3:  					// LPS-402
		  strcpy(HWName, sVNX3);
		  HWMinPhase = 0;
		  HWMaxPhase = 360;
		  HWPhaseStep = 1;
		  HWFMinimum = 20000;		// 2000 MHz
		  HWFMaximum = 40000;		// 4000 MHz
		  break;	  
		case 4:						// LPS-202
		  strcpy(HWName, sVNX4);
		  HWMinPhase = 0;
		  HWMaxPhase = 360;
		  HWPhaseStep = 1;
		  HWFMinimum = 10000;		// 1000 MHz
		  HWFMaximum = 20000;		// 2000 MHz
		  break;
		} /* HWType switch */
	
		// find an open slot to save the data
		// lets see if we have this unit in our table of devices already
		bFound = FALSE;
	
		for (i = 1; i<MAXDEVICES; i++){
		  if (lps[i].SerialNumber == atoi(HWSerial)) {
	        // we already have the device in our table
	        bFound = TRUE;
	        lps[i].DevStatus = lps[i].DevStatus | DEV_CONNECTED; // its here, mark it as connected
	        // at this point the device is present, but not in use, no sense looking more
	        break;
		  }
		}	// end of for loop through the table
	
		// if the device isn't in the table we need to add it
		if (!bFound) {
		  hTemp = 0;
		  for (i = 1; i<MAXDEVICES; i++) {
			if (lps[i].SerialNumber == 0) {
			  hTemp = i;
			  break;
			}
		  } // end of for loop search for an empty slot in our array of devices
	  
		  // save all of the data we've already acquired
		  if (hTemp) {
			lps[hTemp].SerialNumber = atoi(HWSerial);		        // save the device's serial number
			lps[hTemp].DevStatus = lps[hTemp].DevStatus | DEV_CONNECTED; 	// mark it as present
			strcpy (lps[hTemp].ModelName, HWName);     		    	// save the device's model name

			for (i = 1; i < VNX_PROFILE_LENGTH; i++) lps[hTemp].Profile[i] = -1;	// init our profile to "unknown"
			if (lps[hTemp].SerialNumber > 8400) lps[hTemp].NeedsProfileFix = false;
			  else lps[hTemp].NeedsProfileFix = true;				// we need to protect element 19 of the profile from
																			// possible corruption in old units	
			lps[hTemp].MinPhase = HWMinPhase;
			lps[hTemp].MaxPhase = HWMaxPhase;				// default values for phase shift range
			lps[hTemp].MinPhaseStep = HWPhaseStep;				// and resolution
			lps[hTemp].MinFrequency = HWFMinimum;
			lps[hTemp].MaxFrequency = HWFMaximum;				// the frequency range supported by this HW
		  
			// The device has been closed so let's make sure we can find it again
			lps[hTemp].idVendor = dev->descriptor.idVendor;
			lps[hTemp].idProduct = dev->descriptor.idProduct;
			lps[hTemp].idType = HWType;
			strcpy(lps[hTemp].Serialstr, rcvbuff);
		
			if (DEBUG_OUT > 1) {
			  printf("Stored as new device #%d\r\n", hTemp);
			  printf("Serial number=%d\r\n", lps[hTemp].SerialNumber);
			  printf("DevStatus=%08x\r\n", lps[hTemp].DevStatus);
			  printf("Model name=%s\r\n", lps[hTemp].ModelName);
			  printf("MinFrequency=%d\r\n", lps[hTemp].MinFrequency);
			  printf("MaxFrequency=%d\r\n", lps[hTemp].MaxFrequency);
			  printf("Vendor ID=%04x\r\n", lps[hTemp].idVendor);
			  printf("Product ID=%04x\r\n", lps[hTemp].idProduct);
			  printf("Serial number=%s\r\n", lps[hTemp].Serialstr);
			}
		  }
		  else {
	      // our table of devices is full, not much we can do
		  }
		} /* if !bfound  */
		// get any other data we might need
      } // if HWType
    } // for dev
  } // for bus

  // clean up the structure and mark unused slots
  for (i = 1; i<MAXDEVICES; i++){
    if ((lps[i].SerialNumber != 0) && !(lps[i].DevStatus & DEV_CONNECTED))
      lps[i].SerialNumber = 0;	// mark this slot as unused 	

    if (lps[i].SerialNumber == 0)
      lps[i].DevStatus = 0;		// clear the status for robustness!
  }	// end of zombie removal for loop
}

/* ----------------------------------------------------------------- */

void fnLPS_Init(void) {
  /* clear out the storage structure. Must be called once before anything else */
  int i;
  int status;

  for (i = 0; i<MAXDEVICES; i++){
    lps[i].DevStatus = 0; 		// init to no devices connected
    lps[i].SerialNumber = 0; 	// clear the serial number
    lps[i].ModelName[0] = 0;	// put a null string in each model name field
  }

  usb_init();
  if (DEBUG_OUT > 0)  printf("library version %s\r\n", fnLPS_LibVersion());
}

void fnLPS_SetTestMode(bool testmode) {
  TestMode = testmode;
}

int fnLPS_GetNumDevices() {
  int retval = 0;
  int NumDevices = 0;
  int i;
  
  // See how many devices we can find, or have found before
  if (TestMode){
    // construct a fake device
    lps[1].SerialNumber = 66802;
    lps[1].DevStatus = lps[1].DevStatus | DEV_CONNECTED;
	lps[1].idType = 1;
    lps[1].MinPhase = 0;			// 0 degrees is always the minimum
    lps[1].MaxPhase = 360;			// 360 degrees maximum
	lps[1].MinPhaseStep = 1;		// 1 degree units
	lps[1].PhaseAngle = 0;			// start with phase angle = 0
	lps[1].MinFrequency = 40000;	// LPS-802 operates from 4 to 8 GHz
	lps[1].MaxFrequency = 80000;	// units are 100KHz
    strcpy (lps[1].ModelName, "LPS-802");

    // construct a second fake device
    lps[2].SerialNumber = 66123;
    lps[2].DevStatus = lps[2].DevStatus | DEV_CONNECTED;
	lps[2].idType = 2;
    lps[2].MinPhase = 0;			// 0 degrees is always the minimum
    lps[2].MaxPhase = 360;			// 360 degrees maximum
	lps[2].MinPhaseStep = 1;		// 1 degree units
	lps[2].PhaseAngle = 0;			// start with phase angle = 0
	lps[2].MinFrequency = 80000;	// LPS-123 operates from 8 to 12 GHz
	lps[2].MaxFrequency = 120000;	// units are 100KHz
    strcpy (lps[2].ModelName, "LPS-123");

    retval = 2;
  }
  else {
    // go look for some real hardware
    FindVNXDevices();

    // Total up the number of devices we have
    for (i = 0; i < MAXDEVICES; i++){
      if (lps[i].DevStatus & DEV_CONNECTED) NumDevices++; 
    }
    retval = NumDevices;
  }
  return retval;
}

int fnLPS_GetDevInfo(DEVID *ActiveDevices) {
  int i;
  int NumDevices = 0;
  
  if ( ActiveDevices == NULL) return 0; 	// bad array pointer, no place to put the DEVIDs
  
  for (i = 1; i < MAXDEVICES; i++){ 		// NB -- we never put an active device in lps[0] - so DEVIDs start at 1
    if (lps[i].DevStatus & DEV_CONNECTED) {
      ActiveDevices[NumDevices] = i;
      NumDevices++;
    }
  }
  return NumDevices;
}

int fnLPS_GetModelName(DEVID deviceID, char *ModelName) {
  int NumChars = 0;

  if ((deviceID >= MAXDEVICES)  || (deviceID == 0)){
    return 0;
  }

  NumChars = strlen(lps[deviceID].ModelName);
  // If NULL result pointer, just return the number of chars in the name
  if ( ModelName == NULL) return NumChars;
  strcpy(ModelName, lps[deviceID].ModelName);

  return NumChars;
}

LPSTATUS fnLPS_InitDevice(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0)) {
    return INVALID_DEVID;
  }
  
  if (TestMode)
    lps[deviceID].DevStatus = lps[deviceID].DevStatus | DEV_OPENED;
  else {
    // Go ahead and open a handle to the hardware
    if (VNXOpenDevice(deviceID))	//VNXOpenDevice returns 0 if the open succeeded
      return DEVICE_NOT_READY;
    if (DEBUG_OUT > 0) printf("Time to start getting parameters from device %d\r\n", deviceID);

	// Before anything else we need to save profile element 19 for old units
	if (lps[deviceID].NeedsProfileFix) {
	  if (!GetProfileElement(deviceID, 19)) return BAD_HID_IO;
	}
	
    // Get the rest of the device parameters from the device
    if (!GetPhaseAngle(deviceID)) return BAD_HID_IO;
	if (!GetFrequency(deviceID)) return BAD_HID_IO;
	if (!GetRampStart(deviceID)) return BAD_HID_IO;
	if (!GetRampStep(deviceID)) return BAD_HID_IO;
	if (!GetRampStep2(deviceID)) return BAD_HID_IO;
	if (!GetRampEnd(deviceID)) return BAD_HID_IO;
	if (!GetDwellTime(deviceID)) return BAD_HID_IO;
	if (!GetDwellTime2(deviceID)) return BAD_HID_IO;
	if (!GetIdleTime(deviceID)) return BAD_HID_IO;
	if (!GetHoldTime(deviceID)) return BAD_HID_IO;

	// -- get the profile related parameters
	if (!GetProfileIdleTime(deviceID)) return BAD_HID_IO;
	if (!GetProfileDwellTime(deviceID)) return BAD_HID_IO;
	if (!GetProfileCount(deviceID)) return BAD_HID_IO;

  } // end of real device open process case

  // if we got here everything worked OK
  return STATUS_OK;
}

LPSTATUS fnLPS_CloseDevice(DEVID deviceID) {
  
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    lps[deviceID].DevStatus = lps[deviceID].DevStatus & ~DEV_OPENED;
  else {

    // Go ahead and close this hardware - the first step is to stop its read thread
    lps[deviceID].thread_command = THREAD_EXIT;
    
    // The thread handler will close the device. We'll wait up to 1 second then give up.
    int retries;
    retries = 10;
    while (retries && (lps[deviceID].thread_command != THREAD_DEAD)) {
      usleep(100000); /* wait 100 ms */
      retries--;
    }
    if (DEBUG_OUT > 0) printf("After telling the thread to close, we have thread_command=%d retries=%d\r\n", lps[deviceID].thread_command, retries);
    lps[deviceID].thread_command = THREAD_IDLE;

    // Mark it closed in our list of devices
    lps[deviceID].DevStatus = lps[deviceID].DevStatus & ~DEV_OPENED;
  }

  return STATUS_OK;
}

int fnLPS_GetSerialNumber(DEVID deviceID) {
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return 0;
  
  return lps[deviceID].SerialNumber;
}

// ----------------- Functions to set parameters -----------------------------

// Set the working frequency value, must be set before selecting a phase angle

LPSTATUS fnLPS_SetWorkingFrequency(DEVID deviceID, int frequency) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	// -- save the old setting in case we run into trouble --
	int old_frequency = lps[deviceID].Frequency;
	
	// -- round off the frequency to the nearest Megahertz --
	int temp_frequency = frequency / 10;

	if ((frequency % 10) < 5 ) frequency = temp_frequency * 10;	// convert back to 100KHz units
	else frequency = (temp_frequency + 1) * 10;					// round up and convert back to 100KHz units

	if ((frequency >= lps[deviceID].MinFrequency) && (frequency <= lps[deviceID].MaxFrequency)){
	
		lps[deviceID].Frequency = frequency;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the phase angle value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].Frequency;

	if (!SendReport(deviceID, VNX_FREQUENCY | VNX_SET, ptr, 4)){

		lps[deviceID].Frequency = old_frequency;
		ProtectProfile(deviceID);
		return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the phase shift value, the phase angle is in 1 degree units
// NB - the HW does not need profile protection for this command

LPSTATUS fnLPS_SetPhaseAngle(DEVID deviceID, int phase)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0)) {
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)) {
		return DEVICE_NOT_READY;
	}

	LockDev(deviceID, TRUE);

	int old_phase = lps[deviceID].PhaseAngle;
	
	if ((phase >= lps[deviceID].MinPhase) && (phase <= lps[deviceID].MaxPhase)){
	
		lps[deviceID].PhaseAngle = phase;
		if (TestMode){
			LockDev(deviceID, FALSE);
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		LockDev(deviceID, FALSE);
		return BAD_PARAMETER;
	}

	// the phase angle value is OK, lets send it to the hardware
	int tmp_phaseangle = lps[deviceID].PhaseAngle;
	unsigned char *ptr = (unsigned char *) &tmp_phaseangle;

	if (!SendReport(deviceID, VNX_PHASEANGLE | VNX_SET, ptr, 4)){

		lps[deviceID].PhaseAngle = old_phase;
		LockDev(deviceID, FALSE);
		return BAD_HID_IO;
	}

	LockDev(deviceID, FALSE);
	return STATUS_OK;
}

// Set the phase angle of the ramp start
LPSTATUS fnLPS_SetRampStart(DEVID deviceID, int rampstart) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_rampstart = lps[deviceID].RampStart;

	if ((rampstart >= lps[deviceID].MinPhase) && (rampstart <= lps[deviceID].MaxPhase)) {
	  lps[deviceID].RampStart = rampstart;
	  if (TestMode){
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	}
	else {
	  return BAD_PARAMETER;
	}
	
	// the ramp start value is OK, lets send it to the hardware
	int tmp_rampstart = lps[deviceID].RampStart;	
	unsigned char *ptr = (unsigned char *) &tmp_rampstart;
	
	if (!SendReport(deviceID, VNX_ASTART | VNX_SET, ptr, 4)){
	  lps[deviceID].RampStart = old_rampstart;
	  ProtectProfile(deviceID);	  
	  return BAD_HID_IO;
	}
	
	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the phase angle at the end of the ramp

LPSTATUS fnLPS_SetRampEnd(DEVID deviceID, int rampstop) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_rampstop = lps[deviceID].RampStop;
	
	if ((rampstop >= lps[deviceID].MinPhase) && (rampstop <= lps[deviceID].MaxPhase)) {
	  lps[deviceID].RampStop = rampstop;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}
	else {
	  return BAD_PARAMETER;
	}
	// the ramp end value is OK, lets send it to the hardware
	int tmp_rampstop = lps[deviceID].RampStop;
	unsigned char *ptr = (unsigned char *) &tmp_rampstop;

	if (!SendReport(deviceID, VNX_ASTOP | VNX_SET, ptr, 4)) {
	  lps[deviceID].RampStop = old_rampstop;
	  ProtectProfile(deviceID);	  
	  return BAD_HID_IO;
	}
	
	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the phase angle step size for the first section of the ramp
LPSTATUS fnLPS_SetPhaseAngleStep(DEVID deviceID, int phasestep) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_phasestep = lps[deviceID].PhaseStep;

	if (phasestep < (lps[deviceID].MaxPhase - lps[deviceID].MinPhase)){
	  lps[deviceID].PhaseStep = phasestep;
	  if (TestMode){
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	}
	else {
	  return BAD_PARAMETER;
	}
	
	// the phase ramp step value is OK, lets send it to the hardware
	int tmp_phasestep = lps[deviceID].PhaseStep;
	unsigned char *ptr = (unsigned char *) &tmp_phasestep;

	if (!SendReport(deviceID, VNX_ASTEP | VNX_SET, ptr, 4)){
	  lps[deviceID].PhaseStep = old_phasestep;
	  ProtectProfile(deviceID);
	  return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the phase angle step size for the second section of the ramp

LPSTATUS fnLPS_SetPhaseAngleStepTwo(DEVID deviceID, int phasestep2)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0)){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)) {
		return DEVICE_NOT_READY;
	}

	int old_phasestep = lps[deviceID].PhaseStep2;

	if (phasestep2 < (lps[deviceID].MaxPhase - lps[deviceID].MinPhase)){
		lps[deviceID].PhaseStep2 = phasestep2;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}
	// the phase ramp step value is OK, lets send it to the hardware
	int tmp_phasestep2 = lps[deviceID].PhaseStep2;
	unsigned char *ptr = (unsigned char *) &tmp_phasestep2;

	if (!SendReport(deviceID, VNX_ASTEP2 | VNX_SET, ptr, 4)){

		lps[deviceID].PhaseStep2 = old_phasestep;
		ProtectProfile(deviceID);
		return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the time to dwell at each step on the first part of the ramp

LPSTATUS fnLPS_SetDwellTime(DEVID deviceID, int dwelltime) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;
	
	int old_dwelltime = lps[deviceID].DwellTime;

	if (dwelltime >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].DwellTime = dwelltime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].DwellTime;

	if (!SendReport(deviceID, VNX_ADWELL | VNX_SET, ptr, 4)) {
	  lps[deviceID].DwellTime = old_dwelltime;
	  ProtectProfile(deviceID);	  
	  return BAD_HID_IO;
	}
	
	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the time to dwell at each step on the second part of the ramp

LPSTATUS fnLPS_SetDwellTimeTwo(DEVID deviceID, int dwelltime2)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0)){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)) {
		return DEVICE_NOT_READY;
	}

	int old_dwelltime = lps[deviceID].DwellTime2;

	if (dwelltime2 >= VNX_MIN_DWELLTIME){

		lps[deviceID].DwellTime2 = dwelltime2;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].DwellTime2;
	
	if (!SendReport(deviceID, VNX_ADWELL2 | VNX_SET, ptr, 4)){
	  lps[deviceID].DwellTime2 = old_dwelltime;
	  ProtectProfile(deviceID);
	  return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}


// Set the time to wait at the end of the ramp before repeating (in continuous mode)

LPSTATUS fnLPS_SetIdleTime(DEVID deviceID, int idletime) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_idletime = lps[deviceID].IdleTime;

	if (idletime >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].IdleTime = idletime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the idle time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].IdleTime;

	if (!SendReport(deviceID, VNX_AIDLE | VNX_SET, ptr, 4)) {
	  lps[deviceID].IdleTime = old_idletime;
	  ProtectProfile(deviceID);	  
	  return BAD_HID_IO;
	}
	
	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the time to wait at the end of the first section of the ramp before starting the second section

LPSTATUS fnLPS_SetHoldTime(DEVID deviceID, int holdtime) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	int old_holdtime = lps[deviceID].HoldTime;

	if (holdtime >= VNX_MIN_DWELLTIME){
	  lps[deviceID].HoldTime = holdtime;
	  if (TestMode){
	  return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	}
	else {
	  return BAD_PARAMETER;
	}

	// the hold time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].HoldTime;
	
	if (!SendReport(deviceID, VNX_AHOLD | VNX_SET, ptr, 4)){
	  lps[deviceID].HoldTime = old_holdtime;
	  ProtectProfile(deviceID);
	  return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}


// This function is not supported by LPS devices

LPSTATUS fnLPS_SetRFOn(DEVID deviceID, bool on) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0)) 
	  return INVALID_DEVID;
	
	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	if (on) return STATUS_OK;
	else return BAD_PARAMETER;
}

// Set the ramp direction -- "up" is TRUE to ramp upwards
LPSTATUS fnLPS_SetRampDirection(DEVID deviceID, bool up) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	if (up)
	  lps[deviceID].Modebits = lps[deviceID].Modebits & ~SWP_DIRECTION;	// ramp or sweep direction up (bit == 0)
	else
	  lps[deviceID].Modebits = lps[deviceID].Modebits | SWP_DIRECTION;	// ramp or sweep direction downwards

	return STATUS_OK;
}

// Set the ramp mode -- mode = TRUE for repeated ramp, FALSE for one time ramp
LPSTATUS fnLPS_SetRampMode(DEVID deviceID, bool mode) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	if (mode) {
	  lps[deviceID].Modebits = lps[deviceID].Modebits | SWP_CONTINUOUS;		// Repeated ramp or sweep
	  lps[deviceID].Modebits = lps[deviceID].Modebits & ~SWP_ONCE;
	} else {
	  lps[deviceID].Modebits = lps[deviceID].Modebits | SWP_ONCE;			// one time ramp or sweep
	  lps[deviceID].Modebits = lps[deviceID].Modebits & ~SWP_CONTINUOUS;
	}

	return STATUS_OK;
}

// Select bidirectional or unidirectional ramp mode
// TRUE for bidirectional ramp, FALSE for unidirectional ramp
LPSTATUS fnLPS_SetRampBidirectional(DEVID deviceID, bool bidir_enable)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	if (bidir_enable) {
		lps[deviceID].Modebits = lps[deviceID].Modebits | SWP_BIDIR;	// bidirectional ramp
	}
	else {
		lps[deviceID].Modebits = lps[deviceID].Modebits & ~SWP_BIDIR;	// unidirectional ramp
	}

	return STATUS_OK;
}

// Start the ramp immediately
LPSTATUS fnLPS_StartRamp(DEVID deviceID, bool go) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	char VNX_ramp[] = {0, 0, 0, 0};

	if (go)
	  VNX_ramp[0] = (char) lps[deviceID].Modebits & MODE_SWEEP;
	else
	  VNX_ramp[0] = 0;

	if (TestMode)
	  return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_ramp, 1))
	  return BAD_HID_IO;

	return STATUS_OK;
}
// set an individual element in the profile
LPSTATUS fnLPS_SetProfileElement(DEVID deviceID, int index, int phaseangle) {
	int old_element;

	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	if (index < 0 || index > (VNX_PROFILE_LENGTH - 1)) {	// check if the index is valid
	  return BAD_PARAMETER;									// the profile index is invalid
	}

	if ((phaseangle >= lps[deviceID].MinPhase) && (phaseangle <= lps[deviceID].MaxPhase)) {
		old_element = lps[deviceID].Profile[index];			// keep cached value in case IO fails
		if (TestMode){
			lps[deviceID].Profile[index] = phaseangle;
			return STATUS_OK;								// in test mode we update our internal variables,
															// but don't talk to the real HW
		}
	}
	else
	{
		return BAD_PARAMETER;		// the phase angle is out of range
	}
	
	// -- our parameters are good, so lets send them to the hardware
	char VNX_command[] = {0, 0, 0, 0, 0};

	VNX_command[4] = (char) index;
	VNX_command[0] = (char) phaseangle;		// low byte of 16 bit phase angle
	VNX_command[1] = (char)(phaseangle >> 8);	// high byte of 16 bit phase angle

	if (!SendReport(deviceID, VNX_SETPROFILE | VNX_SET, VNX_command, 5)){
		lps[deviceID].Profile[index] = old_element;	
		return BAD_HID_IO;
	}
	return STATUS_OK;
}

// Set the time to remain at each phase angle in a profile
LPSTATUS fnLPS_SetProfileDwellTime(DEVID deviceID, int dwelltime) {
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	int old_dwelltime = lps[deviceID].ProfileDwellTime;

	if (dwelltime >= VNX_MIN_DWELLTIME){
		lps[deviceID].ProfileDwellTime = dwelltime;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the profile dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileDwellTime;

	if (!SendReport(deviceID, VNX_PROFILEDWELL | VNX_SET, ptr, 4)){
		lps[deviceID].ProfileDwellTime = old_dwelltime;
		ProtectProfile(deviceID);
		return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the time to wait at the end of a repeating profile
LPSTATUS fnLPS_SetProfileIdleTime(DEVID deviceID, int idletime)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	int old_idletime = lps[deviceID].ProfileIdleTime;

	if (idletime >= VNX_MIN_DWELLTIME){
		lps[deviceID].ProfileIdleTime = idletime;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the profile idle time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileIdleTime;

	if (!SendReport(deviceID, VNX_PROFILEDWELL | VNX_SET, ptr, 4)){
		lps[deviceID].ProfileIdleTime = old_idletime;
		ProtectProfile(deviceID);
		return BAD_HID_IO;
	}

	ProtectProfile(deviceID);		// if necessary restore profile element 19
	return STATUS_OK;
}

// Set the number of elements in the profile
// NB this function does not require profile element protection, the old HW cannot damage the profile for this command
LPSTATUS fnLPS_SetProfileCount(DEVID deviceID, int profilecount)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	int old_count = lps[deviceID].ProfileCount;

	if (profilecount <= VNX_PROFILE_LENGTH && profilecount > 0){
		lps[deviceID].ProfileCount = profilecount;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the profile count value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileCount;
	
	if (!SendReport(deviceID, VNX_PROFILECOUNT | VNX_SET, ptr, 4)){
		lps[deviceID].ProfileCount = old_count;	
		return BAD_HID_IO;
	}

	return STATUS_OK;
}


// Start the profile immediately
LPSTATUS fnLPS_StartProfile(DEVID deviceID, int mode)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	char VNX_command[] = {0, 0, 0, 0};

	if (mode != 0) {	// mode is 1 for a single profile, 2 for a repeating profile
		VNX_command[0] = (char) ((mode & 0x00000003) | STATUS_PROFILE_ACTIVE);	// start the profile
	}
	else {
		VNX_command[0] = 0;		// stop the profile
	}

	if (TestMode){
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_command, 3)) {
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Save the user settings to flash for autonomous operation
LPSTATUS fnLPS_SaveSettings(DEVID deviceID) {
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW

  char VNX_savesettings[] = {0x42, 0x55, 0x31}; //three byte key to unlock the user protection.

  if (!SendReport(deviceID, VNX_SAVEPAR | VNX_SET, VNX_savesettings, 3))
    return BAD_HID_IO;
  
  return STATUS_OK;
}

// ------------- Functions to get parameters --------------------- 

// Get the phase angle setting
int fnLPS_GetPhaseAngle(DEVID deviceID) {
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  return lps[deviceID].PhaseAngle;
}

// Get the working frequency
int fnLPS_GetWorkingFrequency(DEVID deviceID) {
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  return lps[deviceID].Frequency;
}

// Get the ramp start phase angle
int fnLPS_GetRampStart(DEVID deviceID) {
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  return lps[deviceID].RampStart;
}

// Get the phase angle at the end of the first section of the ramp
int fnLPS_GetRampEnd(DEVID deviceID) {
  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].RampStop;
}

// Get the time to dwell at each step along the first section of  the ramp
int fnLPS_GetDwellTime(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].DwellTime;
}

// Get the time to dwell at each step along the second section of  the ramp
int fnLPS_GetDwellTimeTwo(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].DwellTime2;
}

// Get the idle time to wait at the end of the ramp
int fnLPS_GetIdleTime(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].IdleTime;
}

// -- get the length of the hold time between the first and second sections of the ramp
int fnLPS_GetHoldTime(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].HoldTime;
}

// Get the size of the phase step for the first section of the ramp
int fnLPS_GetPhaseAngleStep(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].PhaseStep;
}

// Get the size of the phase step for the second section of the ramp
int fnLPS_GetPhaseAngleStepTwo(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lps[deviceID].PhaseStep2;
}

int fnLPS_GetProfileDwellTime(DEVID deviceID) {

	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	{
		return DEVICE_NOT_READY;
	}

	return lps[deviceID].ProfileDwellTime;
}

int fnLPS_GetProfileIdleTime(DEVID deviceID) {

	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	return lps[deviceID].ProfileIdleTime;
}

int fnLPS_GetProfileCount(DEVID deviceID) {

	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	return lps[deviceID].ProfileCount;
}

int fnLPS_GetProfileElement(DEVID deviceID, int index)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
		return DEVICE_NOT_READY;

	if (index < 0 || index > (VNX_PROFILE_LENGTH - 1))	{	// check if the index is valid
		return BAD_PARAMETER;
	}
	else {
		if (lps[deviceID].Profile[index] != -1) return lps[deviceID].Profile[index]; // we've got a value cached
		else {
			if (!GetProfileElement(deviceID, index )) {
				return BAD_HID_IO;
			}
			return lps[deviceID].Profile[index];		// we read the value from the HW and cached it.
		}
	}

}

int fnLPS_GetProfileIndex(DEVID deviceID) {

	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID)) {
		return DEVICE_NOT_READY;
	}

	return lps[deviceID].ProfileIndex;
}


// Get the state of the RF output - always reports on
int fnLPS_GetRF_On(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

    return 1;
}

// Get the maximum phase shift for this hardware
int fnLPS_GetMaxPhaseShift(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  return lps[deviceID].MaxPhase;
}

// Get the minimum phase shift for this hardware
int fnLPS_GetMinPhaseShift(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  return lps[deviceID].MinPhase;
}

// And the resolution of the phase setting variable in 1 degree units
int fnLPS_GetMinPhaseStep(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0))
    return INVALID_DEVID;

  return lps[deviceID].MinPhaseStep;
}

// Get the minimum working frequency
int fnLPS_GetMinWorkingFrequency(DEVID deviceID)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	return lps[deviceID].MinFrequency;
}

// get the maximum working frequency
int fnLPS_GetMaxWorkingFrequency(DEVID deviceID)
{
	if ((deviceID >= MAXDEVICES) || (deviceID == 0))
		return INVALID_DEVID;

	return lps[deviceID].MaxFrequency;
}

